"""
Date: 09-12-19
Class: CS5310
Assignment: Graph Library
Author: Brandon Rodriguez


Tests for "Basic Graph" class.
"""

# System Imports.
import unittest

# User Class Imports.
from resources import BasicEdge, BasicNode
from resources import BasicGraph


class TestBasicGraph(unittest.TestCase):
    def setUp(self):
        self.test_graph = BasicGraph()

    def test__graph_initialization(self):
        self.assertEqual(self.test_graph.get_all_edges(), {})
        self.assertEqual(self.test_graph.get_all_nodes(), {})
        self.assertEqual(self.test_graph._edge_name_auto_counter, 1)
        self.assertEqual(self.test_graph._node_name_auto_counter, 1)

    #region Private Information Function Tests

    def test___get_node_auto_name(self):
        with self.subTest('No manual entries.'):
            # Test first call.
            self.assertEqual(1, self.test_graph._get_node_auto_name())

            # Test second call.
            self.assertEqual(2, self.test_graph._get_node_auto_name())

            # Test third call.
            self.assertEqual(3, self.test_graph._get_node_auto_name())

            # Works with 1, 2, and 3 calls. Assuming works for all further n calls.

        with self.subTest('Some conflicting manual entries'):

            # Test first call. Should start at 4, due to above subtests.
            self.assertEqual(4, self.test_graph._get_node_auto_name())

            # Manually add entries 5, 6, and 7 to graph.
            self.test_graph.add_node(BasicNode(5))
            self.test_graph.add_node(BasicNode(6))
            self.test_graph.add_node(BasicNode(7))

            # Test that auto_name now provides 8 as the name.
            self.assertEqual(self.test_graph._get_node_auto_name(), 8)

            # Manually add one more.
            self.test_graph.add_node(BasicNode(9))

            # Test that auto_name now provides 10.
            self.assertEqual(self.test_graph._get_node_auto_name(), 10)

            # Assuming works for all further n calls.

    def test___get_edge_auto_name(self):
        with self.subTest('No manual entries.'):
            # Test first call.
            self.assertEqual(1, self.test_graph._get_edge_auto_name())

            # Test second call.
            self.assertEqual(2, self.test_graph._get_edge_auto_name())

            # Test third call.
            self.assertEqual(3, self.test_graph._get_edge_auto_name())

            # Works with 1, 2, and 3 calls. Assuming works for all further n calls.

        with self.subTest('Some conflicting manual entries'):

            # Test first call. Should start at 4, due to above subtests.
            self.assertEqual(4, self.test_graph._get_edge_auto_name())

            # Manually add entries 5, 6, and 7 to graph.
            edge_5 = BasicEdge(5)
            edge_6 = BasicEdge(6)
            edge_7 = BasicEdge(7)
            self.test_graph._edges[edge_5.get_name()] = edge_5
            self.test_graph._edges[edge_6.get_name()] = edge_6
            self.test_graph._edges[edge_7.get_name()] = edge_7

            # Test that auto_name now provides 8 as the name.
            self.assertEqual(self.test_graph._get_edge_auto_name(), 8)

            # Manually add one more.
            edge_9 = BasicEdge(9)
            self.test_graph._edges[edge_9.get_name()] = edge_9

            # Test that auto_name now provides 10.
            self.assertEqual(self.test_graph._get_edge_auto_name(), 10)

            # Assuming works for all further n calls.

    #endregion Private Information Function Tests

    #region Node Information Function Tests

    def test__get_node_count(self):
        # Test with no nodes.
        self.assertEqual(self.test_graph.get_node_count(), 0)

        # Test with one nodes.
        self.test_graph.create_node()
        self.assertEqual(self.test_graph.get_node_count(), 1)

        # Test with two nodes.
        self.test_graph.create_node()
        self.assertEqual(self.test_graph.get_node_count(), 2)

        # Works with 0, 1, and 2 nodes. Assuming works with all further n nodes.

    def test__get_all_nodes(self):
        node_1 = BasicNode(self.test_graph._get_node_auto_name())
        node_2 = BasicNode(self.test_graph._get_node_auto_name())

        # Test with no node.
        self.assertEqual(self.test_graph.get_node_count(), 0)
        self.assertEqual(self.test_graph.get_all_nodes(), {})

        # Test with one nodes.
        self.test_graph.add_node(node_1)
        self.assertEqual(self.test_graph.get_node_count(), 1)
        self.assertEqual(self.test_graph.get_all_nodes(), {
            node_1.get_name(): node_1,
        })

        # Test with two nodes.
        self.test_graph.add_node(node_2)
        self.assertEqual(self.test_graph.get_node_count(), 2)
        self.assertEqual(self.test_graph.get_all_nodes(), {
            node_1.get_name(): node_1,
            node_2.get_name(): node_2,
        })

        # Works with 0, 1, and 2 nodes. Assuming works with all further n nodes.

    def test__check_if_node_in_graph__success(self):
        # Create nodes.
        node_1 = self.test_graph.create_node()
        node_2 = self.test_graph.create_node()
        node_3 = BasicNode('Node 3')

        with self.subTest('Check by Node class.'):
            # Test node 1.
            self.assertTrue(self.test_graph.check_if_node_in_graph(node_1))

            # Test node 2.
            self.assertTrue(self.test_graph.check_if_node_in_graph(node_2))

            # Test node 3.
            self.assertFalse(self.test_graph.check_if_node_in_graph(node_3))

        with self.subTest('Check by Node name.'):
            # Test node 1.
            self.assertTrue(self.test_graph.check_if_node_in_graph(node_1.get_name()))

            # Test node 2.
            self.assertTrue(self.test_graph.check_if_node_in_graph(node_2.get_name()))

            # Test node 3.
            self.assertFalse(self.test_graph.check_if_node_in_graph(node_3.get_name()))

    def test__check_if_node_in_graph__failure(self):
        with self.subTest('Arg is None'):
            with self.assertRaises(AttributeError):
                self.test_graph.check_if_node_in_graph(None)

    def test__get_node__success(self):
        # Create nodes.
        node_1 = self.test_graph.create_node()
        node_2 = self.test_graph.create_node()

        with self.subTest('Get node by Node class.'):
            # Test first node.
            self.assertEqual(self.test_graph.get_node(node_1), node_1)

            # Test second node.
            self.assertEqual(self.test_graph.get_node(node_2), node_2)

        with self.subTest('Get node by Node name.'):
            # Test first node.
            self.assertEqual(self.test_graph.get_node(node_1.get_name()), node_1)

            # Test second node.
            self.assertEqual(self.test_graph.get_node(node_2.get_name()), node_2)

    def test__get_node__failure(self):
        with self.subTest('Arg is None.'):
            with self.assertRaises(AttributeError):
                self.test_graph.get_node(None)

        with self.subTest('Node is not in graph - by Node class.'):
            node_1 = BasicNode(self.test_graph._get_node_auto_name())
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_node(node_1))

        with self.subTest('Node is not in graph - by Node name.'):
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_node('Test'))

    def test__get_node_degree__success(self):
        with self.subTest('By node class.'):
            # Create nodes.
            node_1 = self.test_graph.create_node()
            node_2 = self.test_graph.create_node()
            node_3 = self.test_graph.create_node()

            # Test with no connections.
            self.assertEqual(self.test_graph.get_node_degree(node_1), 0)

            # Test with one connection.
            self.test_graph.connect_nodes(node_1, node_2)
            self.assertEqual(self.test_graph.get_node_degree(node_1), 1)

            # Test with two connections.
            self.test_graph.connect_nodes(node_1, node_3)
            self.assertEqual(self.test_graph.get_node_degree(node_1), 2)

            # Works with 0, 1, and 2 connections. Assuming works with all further n connections.

        with self.subTest('By node name.'):
            # Create nodes.
            node_1 = self.test_graph.create_node()
            node_2 = self.test_graph.create_node()
            node_3 = self.test_graph.create_node()
            node_4 = self.test_graph.create_node()

            # Test with no connections.
            self.assertEqual(self.test_graph.get_node_degree(node_1.get_name()), 0)

            # Test with one connection.
            self.test_graph.connect_nodes(node_1, node_2)
            self.assertEqual(self.test_graph.get_node_degree(node_1.get_name()), 1)

            # Test with two connections.
            self.test_graph.connect_nodes(node_1, node_3)
            self.assertEqual(self.test_graph.get_node_degree(node_1.get_name()), 2)

            # Works with 0, 1, and 2 connections. Assuming works with all further n connections.

    def test__get_node_degree__failure(self):
        with self.subTest('Arg is None.'):
            with self.assertRaises(AttributeError):
                self.test_graph.get_node_degree(None)

        with self.subTest('Node is not in graph - by Node Class.'):
            node_1 = BasicNode(self.test_graph._get_node_auto_name())
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_node_degree(node_1))

        with self.subTest('Node is not in graph - by Node name.'):
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_node_degree('Test'))

    def test__get_edges_connected_to_node__success(self):
        with self.subTest('By node class.'):
            # Create nodes.
            node_1 = self.test_graph.create_node()
            node_2 = self.test_graph.create_node()
            node_3 = self.test_graph.create_node()

            # Test with no connections.
            self.assertEqual(self.test_graph.get_edges_connected_to_node(node_1), {})

            # Test with one connection.
            edge_1 = self.test_graph.connect_nodes(node_1, node_2)
            self.assertEqual(self.test_graph.get_edges_connected_to_node(node_1), {
                edge_1.get_name(): edge_1,
            })

            # Test with two connections.
            edge_2 = self.test_graph.connect_nodes(node_1, node_3)
            self.assertEqual(self.test_graph.get_edges_connected_to_node(node_1), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
            })

            # Works with 0, 1, and 2 connections. Assuming works with all further n connections.

        with self.subTest('By node name.'):
            # Create nodes.
            node_1 = self.test_graph.create_node()
            node_2 = self.test_graph.create_node()
            node_3 = self.test_graph.create_node()

            # Test with no connections.
            self.assertEqual(self.test_graph.get_edges_connected_to_node(node_1), {})

            # Test with one connection.
            edge_1 = self.test_graph.connect_nodes(node_1, node_2)
            self.assertEqual(self.test_graph.get_edges_connected_to_node(node_1), {
                edge_1.get_name(): edge_1,
            })

            # Test with two connections.
            edge_2 = self.test_graph.connect_nodes(node_1, node_3)
            self.assertEqual(self.test_graph.get_edges_connected_to_node(node_1), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
            })

            # Works with 0, 1, and 2 connections. Assuming works with all further n connections.

    def test__get_edges_connected_to_node__failure(self):
        with self.subTest('Arg is None.'):
            with self.assertRaises(AttributeError):
                self.test_graph.get_edges_connected_to_node(None)

        with self.subTest('Node is not in graph - by Node Class.'):
            node_1 = BasicNode(self.test_graph._get_node_auto_name())
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_edges_connected_to_node(node_1))

        with self.subTest('Node is not in graph - by Node name.'):
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_edges_connected_to_node('Test'))

    def test__get_connected_nodes__success(self):
        with self.subTest('By node class.'):
            # Create nodes.
            node_1 = self.test_graph.create_node()
            node_2 = self.test_graph.create_node()
            node_3 = self.test_graph.create_node()

            # Test with no connections.
            self.assertEqual(self.test_graph.get_connected_nodes(node_1), {})

            # Test with one connection.
            self.test_graph.connect_nodes(node_1, node_2)
            self.assertEqual(self.test_graph.get_connected_nodes(node_1), {
                node_2.get_name(): node_2,
            })
            self.assertEqual(self.test_graph.get_connected_nodes(node_2), {
                node_1.get_name(): node_1,
            })

            # Test with two connections.
            self.test_graph.connect_nodes(node_1, node_3)
            self.assertEqual(self.test_graph.get_connected_nodes(node_1), {
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })
            self.assertEqual(self.test_graph.get_connected_nodes(node_3), {
                node_1.get_name(): node_1,
            })

            # Works with 0, 1, and 2 connections. Assuming works with all further n connections.

        with self.subTest('By node name.'):
            # Create nodes.
            node_1 = self.test_graph.create_node()
            node_2 = self.test_graph.create_node()
            node_3 = self.test_graph.create_node()

            # Test with no connections.
            self.assertEqual(self.test_graph.get_connected_nodes(node_1), {})

            # Test with one connection.
            self.test_graph.connect_nodes(node_1, node_2)
            self.assertEqual(self.test_graph.get_connected_nodes(node_1.get_name()), {
                node_2.get_name(): node_2,
            })
            self.assertEqual(self.test_graph.get_connected_nodes(node_2.get_name()), {
                node_1.get_name(): node_1,
            })

            # Test with two connections.
            self.test_graph.connect_nodes(node_1, node_3)

            self.assertEqual(self.test_graph.get_connected_nodes(node_1.get_name()), {
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })
            self.assertEqual(self.test_graph.get_connected_nodes(node_3.get_name()), {
                node_1.get_name(): node_1,
            })

            # Works with 0, 1, and 2 connections. Assuming works with all further n connections.

    def test__get_connected_nodes__failure(self):
        with self.subTest('Arg is None.'):
            with self.assertRaises(AttributeError):
                self.test_graph.get_connected_nodes(None)

        with self.subTest('Node is not in graph - by Node Class.'):
            node_1 = BasicNode(self.test_graph._get_node_auto_name())
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_connected_nodes(node_1))

        with self.subTest('Node is not in graph - by Node name.'):
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_connected_nodes('Test'))

    def test__determine_node_adjacency__success(self):
        # Create nodes.
        node_1 = self.test_graph.create_node()
        node_2 = self.test_graph.create_node()
        node_3 = self.test_graph.create_node()
        node_4 = self.test_graph.create_node()

        # Connect nodes 1, 2, and 4.
        self.test_graph.connect_nodes(node_1, node_2)
        self.test_graph.connect_nodes(node_1, node_4)

        with self.subTest('Test adjacency by Node class.'):
            # Test adjacency of node 2.
            self.assertTrue(self.test_graph.determine_node_adjacency(node_1, node_2))
            self.assertTrue(self.test_graph.determine_node_adjacency(node_2, node_1))

            # Test adjacency of node 3.
            self.assertFalse(self.test_graph.determine_node_adjacency(node_1, node_3))
            self.assertFalse(self.test_graph.determine_node_adjacency(node_3, node_1))

            # Test adjacency of node 4.
            self.assertTrue(self.test_graph.determine_node_adjacency(node_1, node_4))
            self.assertTrue(self.test_graph.determine_node_adjacency(node_4, node_1))

        with self.subTest('Test adjacency by Node name.'):
            # Test adjacency of node 2.
            self.assertTrue(self.test_graph.determine_node_adjacency(node_1.get_name(), node_2.get_name()))
            self.assertTrue(self.test_graph.determine_node_adjacency(node_2.get_name(), node_1.get_name()))

            # Test adjacency of node 3.
            self.assertFalse(self.test_graph.determine_node_adjacency(node_1.get_name(), node_3.get_name()))
            self.assertFalse(self.test_graph.determine_node_adjacency(node_3.get_name(), node_1.get_name()))

            # Test adjacency of node 4.
            self.assertTrue(self.test_graph.determine_node_adjacency(node_1.get_name(), node_4.get_name()))
            self.assertTrue(self.test_graph.determine_node_adjacency(node_4.get_name(), node_1.get_name()))

    def test__determine_node_adjacency__failure(self):
        # Create nodes.
        node_1 = self.test_graph.create_node()
        node_2 = self.test_graph.create_node()

        with self.subTest('Both nodes None.'):
            with self.assertRaises(AttributeError):
                self.test_graph.determine_node_adjacency(None, None)

        with self.subTest('Node_1 is None.'):
            with self.assertRaises(AttributeError):
                self.test_graph.determine_node_adjacency(None, node_2)

        with self.subTest('Node_1 is None.'):
            with self.assertRaises(AttributeError):
                self.test_graph.determine_node_adjacency(node_1, None)

    #endregion Node Information Function Tests

    #region Edge Information Function Tests

    def test__get_edge_count(self):
        # Create edges to add to graph.
        edge_1 = BasicEdge(self.test_graph._get_node_auto_name())
        edge_2 = BasicEdge(self.test_graph._get_node_auto_name())

        # Test with no edges.
        self.assertEqual(self.test_graph.get_edge_count(), 0)

        # Test with one edges.
        self.test_graph._edges[edge_1.get_name()] = edge_1
        self.assertEqual(self.test_graph.get_edge_count(), 1)

        # Test with two edges.
        self.test_graph._edges[edge_2.get_name()] = edge_2
        self.assertEqual(self.test_graph.get_edge_count(), 2)

        # Works with 0, 1, and 2 edges. Assuming works with all further n edges.

    def test__get_all_edges(self):
        # Create edges to add to graph.
        edge_1 = BasicEdge(self.test_graph._get_node_auto_name())
        edge_2 = BasicEdge(self.test_graph._get_node_auto_name())

        # Test with no edges.
        self.assertEqual(self.test_graph.get_all_edges(), {})

        # Test with one edge.
        self.test_graph._edges[edge_1.get_name()] = edge_1
        self.assertEqual(self.test_graph.get_all_edges(), {
            edge_1.get_name(): edge_1,
        })

        # Test with two edges.
        self.test_graph._edges[edge_2.get_name()] = edge_2
        self.assertEqual(self.test_graph.get_all_edges(), {
            edge_1.get_name(): edge_1,
            edge_2.get_name(): edge_2,
        })

        # Works with 0, 1, and 2 edges. Assuming works with all further n edges.

    def test__check_if_edge_in_graph__success(self):
        # Create nodes.
        node_1 = self.test_graph.create_node()
        node_2 = self.test_graph.create_node()
        node_3 = self.test_graph.create_node()
        node_4 = BasicNode('Node 4')
        node_5 = BasicNode('Node 5')

        # Create connections.
        edge_1 = self.test_graph.connect_nodes(node_1, node_2)
        edge_2 = self.test_graph.connect_nodes(node_1, node_3)
        edge_3 = node_4.connect_node(node_5)

        with self.subTest('Check by Edge class.'):
            # Test edge 1.
            self.assertTrue(self.test_graph.check_if_edge_in_graph(edge_1))

            # Test edge 2.
            self.assertTrue(self.test_graph.check_if_edge_in_graph(edge_2))

            # Test edge 3.
            self.assertFalse(self.test_graph.check_if_edge_in_graph(edge_3))

        with self.subTest('Check by Edge name.'):
            # Test edge 1.
            self.assertTrue(self.test_graph.check_if_edge_in_graph(edge_1.get_name()))

            # Test edge 2.
            self.assertTrue(self.test_graph.check_if_edge_in_graph(edge_2.get_name()))

            # Test edge 3.
            self.assertFalse(self.test_graph.check_if_edge_in_graph(edge_3.get_name()))

    def test__check_if_edge_in_graph__failure(self):
        with self.subTest('Arg is None'):
            with self.assertRaises(AttributeError):
                self.test_graph.check_if_edge_in_graph(None)

    def test__get_edge__success(self):
        # Create and connect nodes.
        node_1 = self.test_graph.create_node()
        node_2 = self.test_graph.create_node()
        node_3 = self.test_graph.create_node()
        edge_1 = self.test_graph.connect_nodes(node_1, node_2)
        edge_2 = self.test_graph.connect_nodes(node_1, node_3)

        with self.subTest('By Edge class.'):
            # Test first edge.
            self.assertEqual(self.test_graph.get_edge(edge_1), edge_1)

            # Test second edge.
            self.assertEqual(self.test_graph.get_edge(edge_2), edge_2)

        with self.subTest('By Edge name.'):
            # Test first edge.
            self.assertEqual(self.test_graph.get_edge(edge_1.get_name()), edge_1)

            # Test second edge.
            self.assertEqual(self.test_graph.get_edge(edge_2.get_name()), edge_2)

    def test__get_edge__failure(self):
        with self.subTest('Arg is None.'):
            with self.assertRaises(AttributeError):
                self.test_graph.get_edge(None)

        with self.subTest('Edge is not in graph - By Edge class.'):
            edge_1 = BasicEdge(self.test_graph._get_edge_auto_name())
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_edge(edge_1))

        with self.subTest('Edge is not in graph - By Edge name.'):
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_edge('Test'))

    def test__get_nodes_connected_to_edge__success(self):
        with self.subTest('By Edge class.'):
            # Create and connect nodes.
            node_1 = self.test_graph.create_node()
            node_2 = self.test_graph.create_node()
            edge_1 = self.test_graph.connect_nodes(node_1, node_2)

            # Test function.
            self.assertEqual(self.test_graph.get_nodes_connected_to_edge(edge_1), (node_1, node_2))
        with self.subTest('By Edge name.'):
            # Create and connect nodes.
            node_1 = self.test_graph.create_node()
            node_2 = self.test_graph.create_node()
            edge_1 = self.test_graph.connect_nodes(node_1, node_2)

            # Test function.
            self.assertEqual(self.test_graph.get_nodes_connected_to_edge(edge_1.get_name()), (node_1, node_2))

    def test__get_nodes_connected_to_edge__failure(self):
        with self.subTest('Arg is None.'):
            with self.assertRaises(AttributeError):
                self.test_graph.get_nodes_connected_to_edge(None)

        with self.subTest('Edge is not in graph - By edge class.'):
            edge_1 = BasicEdge(self.test_graph._get_edge_auto_name())
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_nodes_connected_to_edge(edge_1))

        with self.subTest('Edge is not in graph - By edge name.'):
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_nodes_connected_to_edge('Test'))

    #endregion Edge Information Function Tests

    #region Node Upkeep Function Tests

    def test__create_node(self):
        with self.subTest('With automatic node names.'):
            # Verify start state.
            self.assertEqual(self.test_graph.get_node_count(), 0)
            self.assertEqual(self.test_graph._nodes, {})
            self.assertEqual(len(self.test_graph._nodes), 0)

            # Test with no nodes.
            node_1 = self.test_graph.create_node()
            self.assertTrue(isinstance(node_1, BasicNode))
            self.assertEqual(node_1._graph, self.test_graph)
            self.assertEqual(self.test_graph.get_node_count(), 1)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
            })
            self.assertEqual(node_1.get_name(), '1')

            # Test with one node.
            node_2 = self.test_graph.create_node()
            self.assertTrue(isinstance(node_2, BasicNode))
            self.assertEqual(node_2._graph, self.test_graph)
            self.assertEqual(self.test_graph.get_node_count(), 2)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
            })
            self.assertEqual(node_2.get_name(), '2')

            # Test with two nodes.
            node_3 = self.test_graph.create_node()
            self.assertTrue(isinstance(node_3, BasicNode))
            self.assertEqual(node_3._graph, self.test_graph)
            self.assertEqual(self.test_graph.get_node_count(), 3)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })
            self.assertEqual(node_3.get_name(), '3')

            # Works with 0, 1, and 2 nodes. Assuming works with all further n nodes.

            # Reset for next subtest.
            self.test_graph.remove_node(node_1)
            self.test_graph.remove_node(node_2)
            self.test_graph.remove_node(node_3)
            self.test_graph._edge_name_auto_counter = 0
            self.test_graph._node_name_auto_counter = 0

        with self.subTest('With manual node names.'):
            # Verify start state.
            self.assertEqual(self.test_graph.get_node_count(), 0)
            self.assertEqual(self.test_graph._nodes, {})
            self.assertEqual(len(self.test_graph._nodes), 0)

            # Test with no nodes.
            node_1 = self.test_graph.create_node(node_name='Node 1')
            self.assertTrue(isinstance(node_1, BasicNode))
            self.assertEqual(node_1._graph, self.test_graph)
            self.assertEqual(self.test_graph.get_node_count(), 1)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
            })
            self.assertEqual(node_1.get_name(), 'Node 1')

            # Test with one node.
            node_2 = self.test_graph.create_node(node_name='Node 2')
            self.assertTrue(isinstance(node_2, BasicNode))
            self.assertEqual(node_2._graph, self.test_graph)
            self.assertEqual(self.test_graph.get_node_count(), 2)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
            })
            self.assertEqual(node_2.get_name(), 'Node 2')

            # Test with two nodes.
            node_3 = self.test_graph.create_node(node_name='Node 3')
            self.assertTrue(isinstance(node_3, BasicNode))
            self.assertEqual(node_3._graph, self.test_graph)
            self.assertEqual(self.test_graph.get_node_count(), 3)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })
            self.assertEqual(node_3.get_name(), 'Node 3')

            # Works with 0, 1, and 2 nodes. Assuming works with all further n nodes.

    def test__add_node__success(self):
        with self.subTest('With basic, unconnected nodes.'):
            # Create nodes to connect.
            node_1 = BasicNode(self.test_graph._get_node_auto_name())
            node_2 = BasicNode(self.test_graph._get_node_auto_name())
            node_3 = BasicNode(self.test_graph._get_node_auto_name())

            # Verify start state.
            self.assertEqual(self.test_graph.get_node_count(), 0)
            self.assertEqual(self.test_graph.get_all_nodes(), {})
            self.assertIsNone(node_1._graph)
            self.assertIsNone(node_2._graph)
            self.assertIsNone(node_3._graph)

            # Test with no nodes.
            self.test_graph.add_node(node_1)
            self.assertEqual(node_1._graph, self.test_graph)
            self.assertEqual(self.test_graph.get_node_count(), 1)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
            })

            # Test with one node.
            self.test_graph.add_node(node_2)
            self.assertEqual(node_2._graph, self.test_graph)
            self.assertEqual(self.test_graph.get_node_count(), 2)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
            })

            # Test with two nodes.
            self.test_graph.add_node(node_3)
            self.assertEqual(node_3._graph, self.test_graph)
            self.assertEqual(self.test_graph.get_node_count(), 3)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })

            # Works with 0, 1, and 2 nodes. Assuming works with all further n nodes.

            # Reset graph for next subtest.
            self.test_graph.remove_node(node_1)
            self.test_graph.remove_node(node_2)
            self.test_graph.remove_node(node_3)

        with self.subTest('With interconnected nodes - None in graph.'):
            # Create nodes to connect.
            node_1 = BasicNode(self.test_graph._get_node_auto_name())
            node_2 = BasicNode(self.test_graph._get_node_auto_name())
            node_3 = BasicNode(self.test_graph._get_node_auto_name())
            node_4 = BasicNode(self.test_graph._get_node_auto_name())
            node_5 = BasicNode(self.test_graph._get_node_auto_name())
            node_6 = BasicNode(self.test_graph._get_node_auto_name())
            node_7 = BasicNode(self.test_graph._get_node_auto_name())
            node_8 = BasicNode(self.test_graph._get_node_auto_name())
            node_9 = BasicNode(self.test_graph._get_node_auto_name())

            # Connect nodes.
            edge_1 = node_1.connect_node(node_2)
            edge_2 = node_3.connect_node(node_4)
            edge_3 = node_3.connect_node(node_5)
            edge_4 = node_6.connect_node(node_7)
            edge_5 = node_6.connect_node(node_8)
            edge_6 = node_6.connect_node(node_9)

            # Verify start state.
            self.assertEqual(self.test_graph.get_node_count(), 0)
            self.assertEqual(self.test_graph.get_all_nodes(), {})
            self.assertEqual(self.test_graph.get_edge_count(), 0)
            self.assertEqual(self.test_graph.get_all_edges(), {})

            # Test adding a node with 1 pre-existing connection.
            self.test_graph.add_node(node_1)
            self.assertEqual(self.test_graph.get_node_count(), 2)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
            })
            self.assertEqual(self.test_graph.get_edge_count(), 1)
            self.assertEqual(self.test_graph.get_all_edges(), {
                edge_1.get_name(): edge_1,
            })

            # Test adding a node with 2 pre-existing connections.
            self.test_graph.add_node(node_3)
            self.assertEqual(self.test_graph.get_node_count(), 5)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
                node_5.get_name(): node_5,
            })
            self.assertEqual(self.test_graph.get_edge_count(), 3)
            self.assertEqual(self.test_graph.get_all_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
                edge_3.get_name(): edge_3,
            })

            # Test adding a node with 3 pre-existing connections.
            self.test_graph.add_node(node_6)
            self.assertEqual(self.test_graph.get_node_count(), 9)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
                node_5.get_name(): node_5,
                node_6.get_name(): node_6,
                node_7.get_name(): node_7,
                node_8.get_name(): node_8,
                node_9.get_name(): node_9,
            })
            self.assertEqual(self.test_graph.get_edge_count(), 6)
            self.assertEqual(self.test_graph.get_all_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
                edge_3.get_name(): edge_3,
                edge_4.get_name(): edge_4,
                edge_5.get_name(): edge_5,
                edge_6.get_name(): edge_6,
            })

            # Works with 0, 1, and 2 pre-existing connections.
            # Assuming works with all further n pre-existing connections.

            # Reset graph for next subtest.
            self.test_graph.remove_node(node_1)
            self.test_graph.remove_node(node_2)
            self.test_graph.remove_node(node_3)
            self.test_graph.remove_node(node_4)
            self.test_graph.remove_node(node_5)
            self.test_graph.remove_node(node_6)
            self.test_graph.remove_node(node_7)
            self.test_graph.remove_node(node_8)
            self.test_graph.remove_node(node_9)

        with self.subTest('With interconnected nodes - Some nodes in graph.'):
            # NOTE: If the graph functions are used, then this case should never happen in practice.
            #       Still accounting for it anyways to try to be flexible.

            # Create nodes to connect.
            node_1 = BasicNode(self.test_graph._get_node_auto_name())
            node_2 = self.test_graph.create_node()
            node_3 = BasicNode(self.test_graph._get_node_auto_name())
            node_4 = self.test_graph.create_node()
            node_5 = BasicNode(self.test_graph._get_node_auto_name())
            node_6 = BasicNode(self.test_graph._get_node_auto_name())
            node_7 = self.test_graph.create_node()
            node_8 = BasicNode(self.test_graph._get_node_auto_name())
            node_9 = self.test_graph.create_node()

            # Connect nodes.
            edge_1 = node_1.connect_node(node_2)
            edge_2 = node_3.connect_node(node_4)
            edge_3 = node_3.connect_node(node_5)
            edge_4 = node_6.connect_node(node_7)
            edge_5 = node_6.connect_node(node_8)
            edge_6 = node_6.connect_node(node_9)

            # Verify start state.
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_2.get_name(): node_2,
                node_4.get_name(): node_4,
                node_7.get_name(): node_7,
                node_9.get_name(): node_9,
            })
            self.assertEqual(self.test_graph.get_edge_count(), 0)
            self.assertEqual(self.test_graph.get_all_edges(), {})

            # Test adding a node with 1 pre-existing connection.
            self.test_graph.add_node(node_1)
            self.assertEqual(self.test_graph.get_node_count(), 5)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_4.get_name(): node_4,
                node_7.get_name(): node_7,
                node_9.get_name(): node_9,
            })
            self.assertEqual(self.test_graph.get_edge_count(), 1)
            self.assertEqual(self.test_graph.get_all_edges(), {
                edge_1.get_name(): edge_1,
            })

            # Test adding a node with 2 pre-existing connections.
            self.test_graph.add_node(node_3)
            self.assertEqual(self.test_graph.get_node_count(), 7)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
                node_5.get_name(): node_5,
                node_7.get_name(): node_7,
                node_9.get_name(): node_9,
            })
            self.assertEqual(self.test_graph.get_edge_count(), 3)
            self.assertEqual(self.test_graph.get_all_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
                edge_3.get_name(): edge_3,
            })

            # Test adding a node with 3 pre-existing connections.
            self.test_graph.add_node(node_6)
            self.assertEqual(self.test_graph.get_node_count(), 9)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
                node_5.get_name(): node_5,
                node_6.get_name(): node_6,
                node_7.get_name(): node_7,
                node_8.get_name(): node_8,
                node_9.get_name(): node_9,
            })
            self.assertEqual(self.test_graph.get_edge_count(), 6)
            self.assertEqual(self.test_graph.get_all_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
                edge_3.get_name(): edge_3,
                edge_4.get_name(): edge_4,
                edge_5.get_name(): edge_5,
                edge_6.get_name(): edge_6,
            })

    def test__add_node__failure(self):
        with self.subTest('Arg is not of type Node.'):
            with self.assertRaises(TypeError):
                self.test_graph.add_node(1234)

        # Add node to graph for following subtests.
        new_node = BasicNode(self.test_graph._get_node_auto_name())
        self.assertEqual(self.test_graph.get_node_count(), 0)

        # Add initial node.
        self.test_graph.add_node(new_node)

        with self.subTest('Node already exists within graph.'):
            # Verify start state.
            self.assertEqual(self.test_graph.get_node_count(), 1)

            with self.assertLogs(level='WARNING'):
                # Attempt to re-add same node. Verify that count did not change.
                self.test_graph.add_node(new_node)
                self.assertEqual(self.test_graph.get_node_count(), 1)

    def test__add_node_list__success(self):
        # Create nodes to add.
        node_1 = BasicNode(self.test_graph._get_node_auto_name())
        node_2 = BasicNode(self.test_graph._get_node_auto_name())
        node_3 = BasicNode(self.test_graph._get_node_auto_name())
        node_4 = BasicNode(self.test_graph._get_node_auto_name())
        node_5 = BasicNode(self.test_graph._get_node_auto_name())
        node_6 = BasicNode(self.test_graph._get_node_auto_name())

        with self.subTest('List is type list.'):
            # Create lists.
            list_1 = [node_1]
            list_2 = [node_2, node_3]
            list_3 = [node_4, node_5, node_6]

            # Verify start state.
            self.assertEqual(self.test_graph.get_node_count(), 0)
            self.assertEqual(self.test_graph.get_all_nodes(), {})

            # Test adding a list of 1 item.
            self.test_graph.add_node_list(list_1)
            self.assertEqual(self.test_graph.get_node_count(), 1)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
            })

            # Test adding a list of 2 items.
            self.test_graph.add_node_list(list_2)
            self.assertEqual(self.test_graph.get_node_count(), 3)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })

            # Test adding a list of 3 items.
            self.test_graph.add_node_list(list_3)
            self.assertEqual(self.test_graph.get_node_count(), 6)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
                node_5.get_name(): node_5,
                node_6.get_name(): node_6,
            })

            # Works with lists of size 1, 2, and 3s. Assuming works with all further n-sized lists.

            # Reset graph for next subtest.
            self.test_graph.remove_node(node_1)
            self.test_graph.remove_node(node_2)
            self.test_graph.remove_node(node_3)
            self.test_graph.remove_node(node_4)
            self.test_graph.remove_node(node_5)
            self.test_graph.remove_node(node_6)

        with self.subTest('List is tuple type.'):
            # Create tuples.
            list_1 = (node_1,)
            list_2 = (node_2, node_3,)
            list_3 = (node_4, node_5, node_6,)

            # Verify start state.
            self.assertEqual(self.test_graph.get_node_count(), 0)
            self.assertEqual(self.test_graph.get_all_nodes(), {})

            # Test adding a list of 1 item.
            self.test_graph.add_node_list(list_1)
            self.assertEqual(self.test_graph.get_node_count(), 1)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
            })

            # Test adding a list of 2 items.
            self.test_graph.add_node_list(list_2)
            self.assertEqual(self.test_graph.get_node_count(), 3)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })

            # Test adding a list of 3 items.
            self.test_graph.add_node_list(list_3)
            self.assertEqual(self.test_graph.get_node_count(), 6)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
                node_5.get_name(): node_5,
                node_6.get_name(): node_6,
            })

            # Works with lists of size 1, 2, and 3s. Assuming works with all further n-sized lists.

            # Reset graph for next subtest.
            self.test_graph.remove_node(node_1)
            self.test_graph.remove_node(node_2)
            self.test_graph.remove_node(node_3)
            self.test_graph.remove_node(node_4)
            self.test_graph.remove_node(node_5)
            self.test_graph.remove_node(node_6)

        with self.subTest('List is dict type.'):
            # Create dictionaries.
            list_1 = {node_1.get_name(): node_1}
            list_2 = {node_2.get_name(): node_2, node_3.get_name(): node_3}
            list_3 = {node_4.get_name(): node_4, node_5.get_name(): node_5, node_6.get_name(): node_6}

            # Verify start state.
            self.assertEqual(self.test_graph.get_node_count(), 0)
            self.assertEqual(self.test_graph.get_all_nodes(), {})

            # Test adding a list of 1 item.
            self.test_graph.add_node_list(list_1)
            self.assertEqual(self.test_graph.get_node_count(), 1)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
            })

            # Test adding a list of 2 items.
            self.test_graph.add_node_list(list_2)
            self.assertEqual(self.test_graph.get_node_count(), 3)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })

            # Test adding a list of 3 items.
            self.test_graph.add_node_list(list_3)
            self.assertEqual(self.test_graph.get_node_count(), 6)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
                node_5.get_name(): node_5,
                node_6.get_name(): node_6,
            })

            # Works with lists of size 1, 2, and 3s. Assuming works with all further n-sized lists.

            # Reset graph for next subtest.
            self.test_graph.remove_node(node_1)
            self.test_graph.remove_node(node_2)
            self.test_graph.remove_node(node_3)
            self.test_graph.remove_node(node_4)
            self.test_graph.remove_node(node_5)
            self.test_graph.remove_node(node_6)

    def test__add_node_list__failure(self):
        node_1 = BasicNode(self.test_graph._get_node_auto_name())

        with self.subTest('Arg is not of type list, tuple, or dict.'):
            with self.assertRaises(TypeError):
                self.test_graph.add_node_list(1234)

        with self.subTest('At least one item in list is None'):
            # Verify starting node count.
            self.assertEqual(self.test_graph.get_node_count(), 0)

            # Test None is only item in list.
            with self.assertRaises(AttributeError):
                self.test_graph.add_node_list([None])

            # Test None is first item in list.
            with self.assertRaises(AttributeError):
                self.test_graph.add_node_list([None, node_1])

            # Test None is last item in list.
            with self.assertRaises(AttributeError):
                self.test_graph.add_node_list([node_1, None])

            # Verify node count did not change.
            self.assertEqual(self.test_graph.get_node_count(), 0)

        with self.subTest('At least one item in list is already in graph - by Node class.'):
            node_2 = self.test_graph.create_node()

            # Verify starting node count.
            self.assertEqual(self.test_graph.get_node_count(), 1)

            # Test existing node is only item in list.
            with self.assertRaises(ValueError):
                self.test_graph.add_node_list([node_2])

            # Test existing node is first item in list.
            with self.assertRaises(ValueError):
                self.test_graph.add_node_list([node_2, node_1])

            # Test existing node is last item in list.
            with self.assertRaises(ValueError):
                self.test_graph.add_node_list([node_1, node_2])

            # Verify node count did not change.
            self.assertEqual(self.test_graph.get_node_count(), 1)

        with self.subTest('At least one item in list is already in graph - by Node name.'):
            # Verify starting node count.
            self.assertEqual(self.test_graph.get_node_count(), 1)

            # Test existing node is only item in list.
            with self.assertRaises(ValueError):
                self.test_graph.add_node_list([node_2.get_name()])

            # Test existing node is first item in list.
            with self.assertRaises(ValueError):
                self.test_graph.add_node_list([node_2.get_name(), node_1.get_name()])

            # Test existing node is last item in list.
            with self.assertRaises(ValueError):
                self.test_graph.add_node_list([node_1.get_name(), node_2.get_name()])

            # Verify node count did not change.
            self.assertEqual(self.test_graph.get_node_count(), 1)

    def test__remove_node__success(self):
        with self.subTest('With basic, unconnected nodes.'):
            # Create nodes.
            node_1 = self.test_graph.create_node()
            node_2 = self.test_graph.create_node()
            node_3 = self.test_graph.create_node()

            # Verify start state.
            self.assertEqual(self.test_graph.get_node_count(), 3)
            self.assertEqual(self.test_graph._nodes, {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })

            # Test removal with 3 nodes.
            removed_node_1 = self.test_graph.remove_node(node_3)
            self.assertEqual(node_3, removed_node_1)
            self.assertEqual(self.test_graph.get_node_count(), 2)
            self.assertEqual(self.test_graph._nodes, {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
            })

            # Test removal with 2 nodes.
            removed_node_2 = self.test_graph.remove_node(node_2)
            self.assertEqual(node_2, removed_node_2)
            self.assertEqual(self.test_graph.get_node_count(), 1)
            self.assertEqual(self.test_graph._nodes, {
                node_1.get_name(): node_1,
            })

            # Test removal with 1 node.
            removed_node_3 = self.test_graph.remove_node(node_1)
            self.assertEqual(node_1, removed_node_3)
            self.assertEqual(self.test_graph.get_node_count(), 0)
            self.assertEqual(self.test_graph._nodes, {})

            # Works with 1, 2, and 3 nodes. Assuming works with all further n nodes.

        with self.subTest('With interconnected nodes.'):
            # Create nodes.
            node_1 = self.test_graph.create_node()
            node_2 = self.test_graph.create_node()
            node_3 = self.test_graph.create_node()
            node_4 = self.test_graph.create_node()

            # Create connections.
            edge_1 = self.test_graph.connect_nodes(node_1, node_2)
            edge_2 = self.test_graph.connect_nodes(node_1, node_3)
            edge_3 = self.test_graph.connect_nodes(node_1, node_4)
            edge_4 = self.test_graph.connect_nodes(node_2, node_3)
            edge_5 = self.test_graph.connect_nodes(node_2, node_4)
            edge_6 = self.test_graph.connect_nodes(node_3, node_4)

            # Verify start state.
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_1.get_name(): node_1,
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
            })
            self.assertEqual(self.test_graph.get_edge_count(), 6)
            self.assertEqual(self.test_graph.get_all_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
                edge_3.get_name(): edge_3,
                edge_4.get_name(): edge_4,
                edge_5.get_name(): edge_5,
                edge_6.get_name(): edge_6,
            })
            self.assertEqual(node_1.get_edge_count(), 3)
            self.assertEqual(node_2.get_edge_count(), 3)
            self.assertEqual(node_3.get_edge_count(), 3)
            self.assertEqual(node_4.get_edge_count(), 3)
            self.assertEqual(node_1.get_connected_node_count(), 3)
            self.assertEqual(node_2.get_connected_node_count(), 3)
            self.assertEqual(node_3.get_connected_node_count(), 3)
            self.assertEqual(node_4.get_connected_node_count(), 3)

            # Test removal when node has 3 connections.
            self.test_graph.remove_node(node_1)
            self.assertEqual(self.test_graph.get_node_count(), 3)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
            })
            self.assertEqual(self.test_graph.get_edge_count(), 3)
            self.assertEqual(self.test_graph.get_all_edges(), {
                edge_4.get_name(): edge_4,
                edge_5.get_name(): edge_5,
                edge_6.get_name(): edge_6,
            })
            self.assertEqual(node_1.get_edge_count(), 0)
            self.assertEqual(node_2.get_edge_count(), 2)
            self.assertEqual(node_3.get_edge_count(), 2)
            self.assertEqual(node_4.get_edge_count(), 2)
            self.assertEqual(node_1.get_connected_node_count(), 0)
            self.assertEqual(node_2.get_connected_node_count(), 2)
            self.assertEqual(node_3.get_connected_node_count(), 2)
            self.assertEqual(node_4.get_connected_node_count(), 2)

            # Test removal when node has 2 connections.
            self.test_graph.remove_node(node_2)
            self.assertEqual(self.test_graph.get_node_count(), 2)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
            })
            self.assertEqual(self.test_graph.get_edge_count(), 1)
            self.assertEqual(self.test_graph.get_all_edges(), {
                edge_6.get_name(): edge_6,
            })
            self.assertEqual(node_1.get_edge_count(), 0)
            self.assertEqual(node_2.get_edge_count(), 0)
            self.assertEqual(node_3.get_edge_count(), 1)
            self.assertEqual(node_4.get_edge_count(), 1)
            self.assertEqual(node_1.get_connected_node_count(), 0)
            self.assertEqual(node_2.get_connected_node_count(), 0)
            self.assertEqual(node_3.get_connected_node_count(), 1)
            self.assertEqual(node_4.get_connected_node_count(), 1)

            # Test removal when node has 1 connections.
            self.test_graph.remove_node(node_3)
            self.assertEqual(self.test_graph.get_node_count(), 1)
            self.assertEqual(self.test_graph.get_all_nodes(), {
                node_4.get_name(): node_4,
            })
            self.assertEqual(self.test_graph.get_edge_count(), 0)
            self.assertEqual(self.test_graph.get_all_edges(), {})
            self.assertEqual(node_1.get_edge_count(), 0)
            self.assertEqual(node_2.get_edge_count(), 0)
            self.assertEqual(node_3.get_edge_count(), 0)
            self.assertEqual(node_4.get_edge_count(), 0)
            self.assertEqual(node_1.get_connected_node_count(), 0)
            self.assertEqual(node_2.get_connected_node_count(), 0)
            self.assertEqual(node_3.get_connected_node_count(), 0)
            self.assertEqual(node_4.get_connected_node_count(), 0)

            # Test removal when node has 0 connections.
            self.test_graph.remove_node(node_4)
            self.assertEqual(self.test_graph.get_node_count(), 0)
            self.assertEqual(self.test_graph.get_all_nodes(), {})
            self.assertEqual(self.test_graph.get_edge_count(), 0)
            self.assertEqual(self.test_graph.get_all_edges(), {})
            self.assertEqual(node_1.get_edge_count(), 0)
            self.assertEqual(node_2.get_edge_count(), 0)
            self.assertEqual(node_3.get_edge_count(), 0)
            self.assertEqual(node_4.get_edge_count(), 0)
            self.assertEqual(node_1.get_connected_node_count(), 0)
            self.assertEqual(node_2.get_connected_node_count(), 0)
            self.assertEqual(node_3.get_connected_node_count(), 0)
            self.assertEqual(node_4.get_connected_node_count(), 0)

            # Works with 0, 1, 2, and 3 node connections. Assuming works with all further n node connections.

    def test__remove_node__failure(self):
        with self.subTest('Arg is None'):
            with self.assertRaises(AttributeError):
                self.test_graph.remove_node(None)

        with self.subTest('Node is not in graph - by Node class.'):
            node_1 = BasicNode(self.test_graph._get_node_auto_name())
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_node(node_1))

        with self.subTest('Node is not in graph - by Node name.'):
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.get_node('Test'))

    #endregion Node Upkeep Function Tests

    #region Edge Upkeep Function Tests

    def test__connect_nodes(self):
        # Create nodes to connect.
        node_1 = self.test_graph.create_node()
        node_2 = self.test_graph.create_node()
        node_3 = self.test_graph.create_node()
        node_4 = self.test_graph.create_node()

        # Check that all nodes start with 0 connections.
        self.assertEqual(node_1.get_edge_count(), 0)
        self.assertEqual(node_2.get_edge_count(), 0)
        self.assertEqual(node_3.get_edge_count(), 0)
        self.assertEqual(node_4.get_edge_count(), 0)

        # Test with 0 connections.
        edge_1 = self.test_graph.connect_nodes(node_1, node_2)
        # Check node connection count.
        self.assertEqual(node_1.get_edge_count(), 1)
        self.assertEqual(node_2.get_edge_count(), 1)
        self.assertEqual(node_3.get_edge_count(), 0)
        self.assertEqual(node_4.get_edge_count(), 0)
        # Check connection that was made.
        self.assertTrue(isinstance(edge_1, BasicEdge))
        connected_node_1, connected_node_2 = edge_1.get_nodes()
        self.assertEqual(connected_node_1, node_1)
        self.assertEqual(connected_node_2, node_2)

        # Test with 1 connection.
        edge_2 = self.test_graph.connect_nodes(node_1, node_3)
        # Check node connection count.
        self.assertEqual(node_1.get_edge_count(), 2)
        self.assertEqual(node_2.get_edge_count(), 1)
        self.assertEqual(node_3.get_edge_count(), 1)
        self.assertEqual(node_4.get_edge_count(), 0)
        # Check connection that was made.
        self.assertTrue(isinstance(edge_2, BasicEdge))
        connected_node_1, connected_node_3 = edge_2.get_nodes()
        self.assertEqual(connected_node_1, node_1)
        self.assertEqual(connected_node_3, node_3)

        # Test with 2 connections.
        edge_3 = self.test_graph.connect_nodes(node_1, node_4)
        # Check node connection count.
        self.assertEqual(node_1.get_edge_count(), 3)
        self.assertEqual(node_2.get_edge_count(), 1)
        self.assertEqual(node_3.get_edge_count(), 1)
        self.assertEqual(node_4.get_edge_count(), 1)
        # Check connection that was made.
        self.assertTrue(isinstance(edge_3, BasicEdge))
        connected_node_1, connected_node_4 = edge_3.get_nodes()
        self.assertEqual(connected_node_1, node_1)
        self.assertEqual(connected_node_4, node_4)

        # Works with 0, 1, and 2 connections. Assuming works with all further n connections.

    def test__disconnect_nodes__success(self):
        # Create nodes.
        node_1 = self.test_graph.create_node()
        node_2 = self.test_graph.create_node()
        node_3 = self.test_graph.create_node()
        node_4 = self.test_graph.create_node()

        with self.subTest('Disconnect by node classes.'):
            # Create node connections.
            edge_1 = self.test_graph.connect_nodes(node_1, node_2)
            edge_2 = self.test_graph.connect_nodes(node_1, node_3)
            edge_3 = self.test_graph.connect_nodes(node_1, node_4)

            # Verify start state.
            self.assertEqual(self.test_graph.get_edge_count(), 3)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 3)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
                edge_3.get_name(): edge_3,
            })
            self.assertEqual(node_1.get_connected_node_count(), 3)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
            })

            # Test disconnect with 3 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(node_1_identifier=node_1, node_2_identifier=node_4)
            self.assertEqual(disconnected_edge, edge_3)
            self.assertEqual(self.test_graph.get_edge_count(), 2)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 2)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
            })
            self.assertEqual(node_1.get_connected_node_count(), 2)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })

            # Test disconnect with 2 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(node_1_identifier=node_1, node_2_identifier=node_3)
            self.assertEqual(disconnected_edge, edge_2)
            self.assertEqual(self.test_graph.get_edge_count(), 1)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 1)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
            })
            self.assertEqual(node_1.get_connected_node_count(), 1)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
            })

            # Test disconnect with 1 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(node_1_identifier=node_1, node_2_identifier=node_2)
            self.assertEqual(disconnected_edge, edge_1)
            self.assertEqual(self.test_graph.get_edge_count(), 0)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 0)
            self.assertEqual(node_1.get_edges(), {})
            self.assertEqual(node_1.get_connected_node_count(), 0)
            self.assertEqual(node_1.get_connected_nodes(), {})

            # Works with 1, 2, and 3 connections. Assuming works with all further n connections.

        with self.subTest('Disconnect by node names.'):
            # Create node connections.
            edge_1 = self.test_graph.connect_nodes(node_1, node_2)
            edge_2 = self.test_graph.connect_nodes(node_1, node_3)
            edge_3 = self.test_graph.connect_nodes(node_1, node_4)

            # Verify start state.
            self.assertEqual(self.test_graph.get_edge_count(), 3)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 3)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
                edge_3.get_name(): edge_3,
            })
            self.assertEqual(node_1.get_connected_node_count(), 3)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
            })

            # Test disconnect with 3 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(
                node_1_identifier=node_1.get_name(),
                node_2_identifier=node_4.get_name(),
            )
            self.assertEqual(disconnected_edge, edge_3)
            self.assertEqual(self.test_graph.get_edge_count(), 2)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 2)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
            })
            self.assertEqual(node_1.get_connected_node_count(), 2)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })

            # Test disconnect with 2 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(
                node_1_identifier=node_1.get_name(),
                node_2_identifier=node_3.get_name(),
            )
            self.assertEqual(disconnected_edge, edge_2)
            self.assertEqual(self.test_graph.get_edge_count(), 1)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 1)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
            })
            self.assertEqual(node_1.get_connected_node_count(), 1)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
            })

            # Test disconnect with 1 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(
                node_1_identifier=node_1.get_name(),
                node_2_identifier=node_2.get_name(),
            )
            self.assertEqual(disconnected_edge, edge_1)
            self.assertEqual(self.test_graph.get_edge_count(), 0)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 0)
            self.assertEqual(node_1.get_edges(), {})
            self.assertEqual(node_1.get_connected_node_count(), 0)
            self.assertEqual(node_1.get_connected_nodes(), {})

            # Works with 1, 2, and 3 connections. Assuming works with all further n connections.

        with self.subTest('Disconnect by edge classes.'):
            # Create node connections.
            edge_1 = self.test_graph.connect_nodes(node_1, node_2)
            edge_2 = self.test_graph.connect_nodes(node_1, node_3)
            edge_3 = self.test_graph.connect_nodes(node_1, node_4)

            # Verify start state.
            self.assertEqual(self.test_graph.get_edge_count(), 3)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 3)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
                edge_3.get_name(): edge_3,
            })
            self.assertEqual(node_1.get_connected_node_count(), 3)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
            })

            # Test disconnect with 3 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(edge_identifier=edge_3)
            self.assertEqual(disconnected_edge, edge_3)
            self.assertEqual(self.test_graph.get_edge_count(), 2)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 2)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
            })
            self.assertEqual(node_1.get_connected_node_count(), 2)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })

            # Test disconnect with 2 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(edge_identifier=edge_2)
            self.assertEqual(disconnected_edge, edge_2)
            self.assertEqual(self.test_graph.get_edge_count(), 1)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 1)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
            })
            self.assertEqual(node_1.get_connected_node_count(), 1)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
            })

            # Test disconnect with 1 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(edge_identifier=edge_1)
            self.assertEqual(disconnected_edge, edge_1)
            self.assertEqual(self.test_graph.get_edge_count(), 0)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 0)
            self.assertEqual(node_1.get_edges(), {})
            self.assertEqual(node_1.get_connected_node_count(), 0)
            self.assertEqual(node_1.get_connected_nodes(), {})

            # Works with 1, 2, and 3 connections. Assuming works with all further n connections.

        with self.subTest('Disconnect by edge names.'):
            # Create node connections.
            edge_1 = self.test_graph.connect_nodes(node_1, node_2)
            edge_2 = self.test_graph.connect_nodes(node_1, node_3)
            edge_3 = self.test_graph.connect_nodes(node_1, node_4)

            # Verify start state.
            self.assertEqual(self.test_graph.get_edge_count(), 3)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 3)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
                edge_3.get_name(): edge_3,
            })
            self.assertEqual(node_1.get_connected_node_count(), 3)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
                node_4.get_name(): node_4,
            })

            # Test disconnect with 3 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(edge_identifier=edge_3.get_name())
            self.assertEqual(disconnected_edge, edge_3)
            self.assertEqual(self.test_graph.get_edge_count(), 2)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 2)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
                edge_2.get_name(): edge_2,
            })
            self.assertEqual(node_1.get_connected_node_count(), 2)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
                node_3.get_name(): node_3,
            })

            # Test disconnect with 2 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(edge_identifier=edge_2.get_name())
            self.assertEqual(disconnected_edge, edge_2)
            self.assertEqual(self.test_graph.get_edge_count(), 1)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 1)
            self.assertEqual(node_1.get_edges(), {
                edge_1.get_name(): edge_1,
            })
            self.assertEqual(node_1.get_connected_node_count(), 1)
            self.assertEqual(node_1.get_connected_nodes(), {
                node_2.get_name(): node_2,
            })

            # Test disconnect with 1 connections.
            disconnected_edge = self.test_graph.disconnect_nodes(edge_identifier=edge_1.get_name())
            self.assertEqual(disconnected_edge, edge_1)
            self.assertEqual(self.test_graph.get_edge_count(), 0)
            self.assertEqual(self.test_graph.get_node_count(), 4)
            self.assertEqual(node_1.get_edge_count(), 0)
            self.assertEqual(node_1.get_edges(), {})
            self.assertEqual(node_1.get_connected_node_count(), 0)
            self.assertEqual(node_1.get_connected_nodes(), {})

            # Works with 1, 2, and 3 connections. Assuming works with all further n connections.

    def test__disconnect_nodes__failure(self):
        with self.subTest('No args passed.'):
            with self.assertRaises(AttributeError):
                self.test_graph.disconnect_nodes()

        with self.subTest('Node and Edge identifiers passed at once.'):
            # Edge and node_1 passed.
            with self.assertRaises(AttributeError):
                self.test_graph.disconnect_nodes(node_2_identifier='1', edge_identifier='2')

            # Edge and node_2 passed.
            with self.assertRaises(AttributeError):
                self.test_graph.disconnect_nodes(node_1_identifier='1', edge_identifier='2')

        # Create nodes for following tests.
        node_1 = BasicNode(self.test_graph._get_node_auto_name())
        node_2 = BasicNode(self.test_graph._get_node_auto_name())
        edge_1 = BasicEdge(self.test_graph._get_edge_auto_name())

        # Add only one node to graph.
        self.test_graph.add_node(node_1)

        with self.subTest('Node args passed but node 1 not in graph.'):
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.disconnect_nodes(node_1_identifier=node_2, node_2_identifier=node_1))

        with self.subTest('Node args passed but node 2 not in graph.'):
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.disconnect_nodes(node_1_identifier=node_1, node_2_identifier=node_2))

        with self.subTest('Node args passed but nodes aren\'t connected.'):
            # Add second node to graph.
            self.test_graph.add_node(node_2)

            # Check for error.
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.disconnect_nodes(edge_identifier=edge_1))

        with self.subTest('Edge arg passed but edge not in graph.'):
            # Connect two nodes in graph.
            self.test_graph.connect_nodes(node_1, node_2)

            # Check for error.
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_graph.disconnect_nodes(edge_identifier=edge_1))

    #endregion Edge Upkeep Function Tests
