"""
Date: 09-12-19
Class: CS5310
Assignment: Graph Library
Author: Brandon Rodriguez


Tests for "Basic Edge" class.
"""

# System Imports.
import unittest

# User Class Imports.
from resources import BasicEdge, BasicNode


class TestBasicEdge(unittest.TestCase):
    def setUp(self):
        self.edge_name = 'Test Edge Name'
        self.node_name = 'Test Node Name'
        self.test_edge = BasicEdge(self.edge_name)

    def test__edge_initialization(self):
        self.assertEqual(self.test_edge.get_name(), self.edge_name)
        self.assertEqual(self.test_edge.get_nodes(), ())
        self.assertEqual(self.test_edge.get_node_count(), 0)

    #region Information Function Tests

    def test__get_name(self):
        self.assertTrue(isinstance(self.test_edge.get_name(), str))
        self.assertEqual(self.test_edge.get_name(), self.edge_name)

    def test__get_node_count(self):
        # Create nodes to connect.
        node_1 = BasicNode('Node 1')
        node_2 = BasicNode('Node 2')

        # Test with no nodes.
        self.assertEqual(self.test_edge.get_node_count(), 0)

        # Test with two nodes.
        self.test_edge.connect_nodes(node_1, node_2)

        self.assertEqual(self.test_edge.get_node_count(), 2)

    def test__get_nodes__success(self):
        # Create nodes to connect.
        node_1 = BasicNode('Node 1')
        node_2 = BasicNode('Node 2')

        # Test with no nodes.
        self.assertEqual(self.test_edge.get_nodes(), ())

        # Test with two nodes.
        self.test_edge.connect_nodes(node_1, node_2)

        self.assertEqual(self.test_edge.get_nodes(), (node_1, node_2))

    def test__get_nodes__failure(self):
        # Create nodes to connect.
        node_1 = BasicNode('Node 1')
        node_2 = BasicNode('Node 2')

        with self.subTest('Failure - Only one node.'):
            with self.assertRaises(AttributeError):
                self.test_edge._nodes.append(node_1)
                self.test_edge.get_nodes()

        with self.subTest('Failure - More than two nodes.'):
            with self.assertRaises(AttributeError):
                self.test_edge.connect_nodes(node_1, node_2)
                self.test_edge._nodes.append(node_1)
                self.test_edge.get_nodes()

    def test__get_node__success(self):
        # Create and connect nodes.
        node_1 = BasicNode('Node 1')
        node_2 = BasicNode('Node 2')
        self.test_edge.connect_nodes(node_1, node_2)

        with self.subTest('Get node by Node class.'):
            # Test first node.
            returned_node = self.test_edge.get_node(node_1)
            self.assertEqual(returned_node, node_1)

            # Test second node.
            returned_node = self.test_edge.get_node(node_2)
            self.assertEqual(returned_node, node_2)

        with self.subTest('Get node by Node name.'):
            # Test first node.
            returned_node = self.test_edge.get_node(node_1.get_name())
            self.assertEqual(returned_node, node_1)

            # Test second node.
            returned_node = self.test_edge.get_node(node_2.get_name())
            self.assertEqual(returned_node, node_2)

    def test__get_node__failure(self):
        with self.subTest('Arg of type None.'):
            with self.assertRaises(AttributeError):
                self.test_edge.get_node(None)

        with self.subTest('Node not connected to edge - by Node class.'):
            node_1 = BasicNode('Node 1')
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_edge.get_node(node_1))

        with self.subTest('Node not connected to edge - by Node name.'):
            with self.assertLogs(level='WARNING'):
                self.assertIsNone(self.test_edge.get_node('Test'))

    def test__get_connected_node__success(self):
        # Create nodes to connect.
        node_1 = BasicNode('Node 1')
        node_2 = BasicNode('Node 2')

        # Connect nodes.
        self.test_edge.connect_nodes(node_1, node_2)

        # Test method.
        self.assertEqual(self.test_edge.get_partner_node(node_1), node_2)
        self.assertEqual(self.test_edge.get_partner_node(node_2), node_1)

    def test__get_connected_node__failure(self):
        # Create nodes to connect.
        node_1 = BasicNode('Node 1')
        node_2 = BasicNode('Node 2')

        with self.subTest('Arg is not of type Node.'):
            with self.assertRaises(TypeError):
                self.test_edge.get_partner_node(1234)

        with self.subTest('No nodes connected.'):
            # Should start with 0 connected nodes.
            self.assertEqual(self.test_edge.get_node_count(), 0)

            with self.assertRaises(AttributeError):
                self.test_edge.get_partner_node(node_1)

            with self.assertRaises(AttributeError):
                self.test_edge.get_partner_node(node_2)

        with self.subTest('Only one node connected.'):
            # Should start with 0 connected nodes from previous subtests.
            self.assertEqual(self.test_edge.get_node_count(), 0)
            self.test_edge._nodes.append(node_1)

            with self.assertRaises(AttributeError):
                self.test_edge.get_partner_node(node_1)

            with self.assertRaises(AttributeError):
                self.test_edge.get_partner_node(node_2)

        with self.subTest('More than two nodes connected.'):
            # Should start with 1 connected node from previous subtests.
            self.assertEqual(self.test_edge.get_node_count(), 1)

            self.test_edge._nodes.append(node_1)
            self.test_edge._nodes.append(node_2)

            with self.assertRaises(AttributeError):
                self.test_edge.get_partner_node(node_1)

            with self.assertRaises(AttributeError):
                self.test_edge.get_partner_node(node_2)

    #endregion Information Function Tests

    #region Upkeep Function Tests

    def test__connect_nodes__success(self):
        # Create nodes to connect.
        node_1 = BasicNode('Node 1')
        node_2 = BasicNode('Node 2')

        # Verify initial state.
        self.assertEqual(self.test_edge.get_node_count(), 0)
        self.assertEqual(self.test_edge._nodes, [])

        # Connect nodes.
        self.test_edge.connect_nodes(node_1, node_2)
        self.assertEqual(self.test_edge.get_node_count(), 2)
        self.assertEqual(self.test_edge._nodes, [node_1, node_2])
        self.assertEqual(node_1.get_edges(), {self.test_edge.get_name(): self.test_edge})
        self.assertEqual(node_2.get_edges(), {self.test_edge.get_name(): self.test_edge})
        self.assertEqual(node_1.get_connected_nodes(), {node_2.get_name(): node_2})

    def test__connect_nodes__failure(self):
        # Create nodes to connect.
        node_1 = BasicNode('Node 1')
        node_2 = BasicNode('Node 2')

        with self.subTest('Arg 1 is not of type Node.'):
            # Should have 0 connected node from above subtests.
            self.assertEqual(self.test_edge.get_node_count(), 0)

            # Check with first arg not type Node.
            with self.assertRaises(TypeError):
                self.test_edge.connect_nodes(1234, node_2)

            # Check that connection count hasn't changed.
            self.assertEqual(self.test_edge.get_node_count(), 0)

        with self.subTest('Arg 2 is not of type Node.'):
            # Should have 0 connected node from above subtests.
            self.assertEqual(self.test_edge.get_node_count(), 0)

            # Test with second arg not type Node.
            with self.assertRaises(TypeError):
                self.test_edge.connect_nodes(node_1, 1234)

            # Check that connection count hasn't changed.
            self.assertEqual(self.test_edge.get_node_count(), 0)

        with self.subTest('Node names are identical.'):
            # Should have 0 connected node from above subtests.
            self.assertEqual(self.test_edge.get_node_count(), 0)

            # Test with duplicate Node names.
            node_1_duplicate = BasicNode(node_1.get_name())
            with self.assertRaises(ValueError):
                self.test_edge.connect_nodes(node_1, node_1_duplicate)

            # Check that connection count hasn't changed.
            self.assertEqual(self.test_edge.get_node_count(), 0)

        with self.subTest('One node already connected.'):
            # Should have 0 connected node from above subtests.
            self.assertEqual(self.test_edge.get_node_count(), 0)

            # Connect exactly one node.
            self.test_edge._nodes.append(node_1)

            # Now try to connect any other pairs.
            with self.assertRaises(AttributeError):
                self.test_edge.connect_nodes(node_1, node_2)
            with self.assertRaises(AttributeError):
                self.test_edge.connect_nodes(node_2, node_1)

            # Should end with 1 connected nodes.
            self.assertEqual(self.test_edge.get_node_count(), 1)

        with self.subTest('More than one connection.'):
            # Should have 1 connected node from above subtests.
            self.assertEqual(self.test_edge.get_node_count(), 1)

            # Connect a second node to get a valid pair.
            self.test_edge._nodes.append(node_2)

            # Now try to connect any other pairs.
            with self.assertRaises(AttributeError):
                self.test_edge.connect_nodes(node_1, node_2)
            with self.assertRaises(AttributeError):
                self.test_edge.connect_nodes(node_2, node_1)

            # Check that connection count hasn't changed.
            self.assertEqual(self.test_edge.get_node_count(), 2)

    def test__connect_node_to_self(self):
        self.test_edge._nodes_can_self_connect = True
        test_node = BasicNode('Test Node')
        self.test_edge.connect_nodes(test_node, test_node)
        self.assertEqual(self.test_edge.get_nodes(), (test_node, test_node))

    def test__disconnect_nodes__success(self):
        # Create and connect nodes.
        node_1 = BasicNode('Node 1')
        node_2 = BasicNode('Node 2')
        self.test_edge.connect_nodes(node_1, node_2)

        # Verify initial state.
        self.assertEqual(self.test_edge.get_node_count(), 2)
        self.assertEqual(self.test_edge.get_nodes(), (node_1, node_2))

        # Test disconnect.
        removed_edge = self.test_edge.disconnect_nodes()
        self.assertEqual(removed_edge, self.test_edge)
        self.assertEqual(self.test_edge.get_node_count(), 0)
        self.assertEqual(self.test_edge.get_nodes(), ())

    def test_disconnect_nodes_failure(self):
        with self.subTest('No nodes connected.'):
            self.assertIsNone(self.test_edge.disconnect_nodes())

    #endregion Upkeep Function Tests
