"""tests"""
from django.test import TestCase
from django.contrib.auth.models import User, Permission
from django.core import mail
from django.conf import settings
from django.core.urlresolvers import reverse

from volunteer.models import VolunteerTopic


class VolunteerTopicTestCase(TestCase):
    """Tests for VolunteerTopic"""
    def _create_users(self):
        self.password = "pass"
        self.email = 'test@example.com'
        self.volunteer_email = 'volunteer-updates@example.com'
        self.users = {}
        users = [
            'volunteer',
            'non_volunteer',
            'volunteer_admin',
        ]
        for user in users:
            self.users[user] = User.objects.create_user(
                user, password=self.password, email=self.email)
        perm = Permission.objects.get(codename='change_volunteertopic')
        self.users['volunteer_admin'].user_permissions.add(perm)
        #self.users.save()

    def setUp(self):
        self._create_users()
        self.topic = VolunteerTopic.objects.create(name="Test Topic",
                                                   content="Topic Content")
        self.topic.volunteers.create(user=self.users['volunteer'],
                                     subscribed=False)

        # Disable limit for testing
        settings.EMAIL_LIMIT_MINUTES = 0

    def test_email_no_subscribers(self):
        """Test no emails are sent if no subscribers"""
        volunteer = self.topic.volunteers.all()[0]
        volunteer.subscribed = False
        volunteer.save()

        length = len(mail.outbox)
        self.topic.save()
        self.assertEqual(len(mail.outbox), length)

    def test_email(self):
        """Test that with subscribers, emails are sent"""
        volunteer = self.topic.volunteers.all()[0]
        volunteer.subscribed = True
        volunteer.save()

        length = len(mail.outbox)

        with self.settings(EMAIL_LIMIT_MINUTES=0,
                           VOLUNTEER_FROM_EMAIL=self.volunteer_email):
            self.topic.save()
        self.assertEqual(len(mail.outbox), length+1)
        self.assertEqual(mail.outbox[length].to, [self.email])

    def test_email_time_limit(self):
        """Test that time limit is used"""
        volunteer = self.topic.volunteers.all()[0]
        volunteer.subscribed = True
        volunteer.save()

        self.topic.save()  # Trigger last_modified

        length = len(mail.outbox)
        with self.settings(EMAIL_LIMIT_MINUTES=5):
            self.topic.save()
        self.assertEqual(len(mail.outbox), length)

    def test_view_topic(self):
        """Test if login returns topic"""
        self.client.login(username=self.users['volunteer'].username,
                          password=self.password)
        url = reverse('volunteer:view_volunteer_topics')
        response = self.client.get(url)
        self.assertEqual(response.status_code, 200,
                         "View did not return a 200.")

    def test_view_topic_not_volunteer(self):
        """Test that non-volunteers see 302 Redirect"""
        self.client.login(username=self.users['non_volunteer'].username,
                          password=self.password)
        url = reverse('volunteer:view_volunteer_topics')
        response = self.client.get(url)
        self.assertEqual(response.status_code, 302,
                         "View did not return a 302.")

    def test_edit_topic(self):
        """Test if login allows edit topic"""
        self.client.force_login(self.users['volunteer_admin'])
        url = reverse('volunteer:edit_volunteer_topic', args=(self.topic.pk,))
        response = self.client.get(url)
        self.assertEqual(response.status_code, 200,
                         "View did not return a 200.")

    def test_edit_topic_non_admin(self):
        """Test that non-admins see 403 Forbidden"""
        self.client.force_login(self.users['volunteer'])
        url = reverse('volunteer:edit_volunteer_topic', args=(self.topic.pk,))
        response = self.client.get(url)
        self.assertEqual(response.status_code, 403,
                         "View did not return a 403.")
        self.client.force_login(self.users['non_volunteer'])
        url = reverse('volunteer:edit_volunteer_topic', args=(self.topic.pk,))
        response = self.client.get(url)
        self.assertEqual(response.status_code, 403,
                         "View did not return a 403.")
