"""forms"""
from django import forms
from django.conf import settings
from django.contrib.auth.models import User
from django.contrib.auth.forms import UserCreationForm
from django.core.validators import validate_email

from volunteer.models import Volunteer, VolunteerTopic, VolunteerCode


class MultiEmailField(forms.Field):
    INVALID_EMAILS = 'Enter valid email addresses.'
    widget = forms.widgets.Textarea

    def to_python(self, value):
        #print('to_python({0!r})'.format(value))
        if not value:
            return []

        return [x.strip() for x in value.splitlines() if x]

    def validate(self, values):
        #print('validate({0!r})'.format(values))
        super(MultiEmailField, self).validate(self)

        try:
            for email in values:
                validate_email(email)
        except forms.ValidationError:
            raise forms.ValidationError(
                MultiEmailField.INVALID_EMAILS, code='invalid')


class VolunteerCodeForm(forms.ModelForm):
    """All fields must be set as required=False because they are not always
    show. They must be checked in the main clean method, if emails is set."""
    subject = forms.CharField(
        required=False,
        initial=getattr(settings, 'VOLUNTEER_EMAIL_SUBJECT', '').strip(),
        help_text='This is the subject of the email.')
    body = forms.CharField(
        required=False,
        widget=forms.widgets.Textarea,
        initial=getattr(settings, 'VOLUNTEER_EMAIL_MESSAGE', '').strip(),
        help_text='This is the body of the email. The following items will automatically be replaced:'
        '<ul>'
        '<li><kbd>{link}</kbd> - The code link.</li>'
        '<li><kbd>{expires}</kbd> - The expiration date.</li>'
        '<li><kbd>{topic_list}</kbd> - A list of selected topic names.</li>'
        '</ul>')
    emails = MultiEmailField(
        required=False,
        help_text='Enter one or more emails each on a different line. '
        'Emails are only sent if this is filled out when saved.',
        label='Volunteer Emails')

    class Meta:
        model = VolunteerCode
        fields = ['expired', 'expires_after', 'topics']

    def clean(self):
        msg = 'The following is required: {0}, because you set {1}.'
        cleaned_data = super(VolunteerCodeForm, self).clean()

        if self.errors:
            return

        subject = cleaned_data.get('subject')
        body = cleaned_data.get('body')
        emails = cleaned_data.get('emails')

        if emails:
            errors = []
            if not subject:
                errors.append('Subject')
            if not body:
                errors.append('Body')

            if errors:
                raise forms.ValidationError(msg.format(
                ', '.join(errors), self.fields['emails'].label))

            msg2 = 'Required tag not used in body: {0!r}.'

            required_tags = ['{link}', '{expires}']

            for tag in required_tags:
                if tag not in body:
                    raise forms.ValidationError(msg2.format(tag))

class VolunteerTopicForm(forms.ModelForm):
    """Simple VolunteerTopic Form"""
    error_css_class = 'error'
    required_css_class = 'required'

    class Meta:
        model = VolunteerTopic
        fields = ['name', 'content']


class CustomUserChoiceField(forms.ModelChoiceField):
    """Override label to show full name"""
    def label_from_instance(self, user):
        full = user.get_full_name() or "<Not Set>"
        return "{0} ({1})".format(full, user)


class VolunteerForm(forms.ModelForm):
    """Use custom field for user"""
    # pylint: disable=no-member
    user = CustomUserChoiceField(queryset=User.objects.all(), label='user')

    class Meta:
        model = Volunteer
        fields = ['user', 'topic', 'subscribed']


class UserForm(UserCreationForm):
    error_css_class = "error"
    required_css_class = "required"

    code = forms.CharField(
        required=True,
        help_text="Enter account creation code you should have recieved.",
        widget=forms.TextInput(attrs={
            'size': '36',
        }))

    class Meta:
        model = User
        fields = ['username', 'email', 'first_name', 'last_name']

    def __init__(self, *args, **kwargs):
        super(UserForm, self).__init__(*args, **kwargs)
        self.fields['username'].help_text = 'This is case sensitive.'
        self.fields['email'].required = True

    def _clean_name(self, key):
        name = self.cleaned_data[key].strip()
        if not name:
            raise forms.ValidationError(
                self.fields[key].error_messages['required'])
        return name

    def clean_first_name(self):
        return self._clean_name('first_name')

    def clean_last_name(self):
        return self._clean_name('last_name')

    def clean_code(self):
        #print("clean_code()")
        code = self.cleaned_data['code']
        user_code = None
        try:
            user_code = VolunteerCode.objects.get(code=code)
        except (VolunteerCode.DoesNotExist, ValueError):
            #print("Does not exist!")
            pass
        if not user_code:
            #print("Raising validation error")
            raise forms.ValidationError("That code is invalid.")
        else:
            if not user_code.valid():
                raise forms.ValidationError("That code has expired.")

        return user_code
