"""Admin"""
from copy import deepcopy

from django.contrib import admin, messages
from django.conf import settings
from django.core.mail import send_mass_mail
from django.core.urlresolvers import reverse
from django.utils import timezone
from django.template.defaulttags import date

from volunteer.models import Volunteer, VolunteerCode
from volunteer.forms import VolunteerForm, VolunteerCodeForm


def make_expired(modeladmin, request, queryset):
    queryset.update(expired=True)
make_expired.short_description = "Expire selected volunteer codes"


class VolunteerInline(admin.TabularInline):
    """Simple Volunteer Inline for VolunteerTopic"""
    model = Volunteer
    extra = 4
    form = VolunteerForm


class VolunteerTopicAdmin(admin.ModelAdmin):
    """Simple VolunteerTopic Admin"""
    list_display = ('name', 'num_volunteers', 'num_subscribed',
                    'last_modified')
    inlines = (VolunteerInline,)
    search_fields = ['name']


class VolunteerCodeAdmin(admin.ModelAdmin):
    """Simple VolunteerCode Admin"""
    model = VolunteerCode
    form = VolunteerCodeForm
    list_display = ('__str__', 'valid', 'date_created', 'expires_after')
    date_hierarchy = 'date_created'
    list_filter = ['expired']
    change_form_template = 'admin/volunteer/change_volunteer_code.html'
    actions = [make_expired]
    filter_horizontal = ['topics']

    fieldsets = (
        (None, {
            'fields': ('topics', 'expires_after')
        }),
        ('Advanced', {
            'classes': ('collapse',),
            'fields': ('expired',)
        }),
    )

    def get_fieldsets(self, request, obj=None):
        "Show the email fields if we have a valid code"
        fieldsets = deepcopy(
            super(VolunteerCodeAdmin, self).get_fieldsets(request, obj))
        if obj and not obj.expired:
            new = fieldsets + (
                ('Email Volunteers', {
                    'fields': ('subject', 'body', 'emails')
                }),
            )
            fieldsets = new

        return fieldsets

    def change_view(self, request, object_id, form_url='', extra_context=None):
        extra_context = extra_context or {}
        form = self.form(request.POST)
        try:
            v_code = VolunteerCode.objects.get(pk=object_id)
            link = '{0}?code={1}'.format(
                reverse('volunteer:new_volunteer'), v_code.code)
            link = request.build_absolute_uri(link)
            extra_context['link'] = link

            if request.method == 'POST':
                self._handle_email_form(request, v_code, link)
        except VolunteerCode.DoesNotExist:
            pass
        return super(VolunteerCodeAdmin, self).change_view(
            request, object_id, form_url, extra_context=extra_context)

    def _handle_email_form(self, request, v_code, link):
        form = self.form(request.POST)
        emails = None
        if form.is_valid():
            emails = form.cleaned_data['emails']

        if not emails:
            return

        expires_after = form.cleaned_data['expires_after']
        expires = v_code.date_created + timezone.timedelta(days=expires_after)

        topics = form.cleaned_data['topics']
        topics = [topic.name for topic in topics]
        topic_list = '\n - '.join(topics)
        if len(topics) > 0:
            topic_list = ' - ' + topic_list

        subject = form.cleaned_data['subject']
        body = form.cleaned_data['body'].format(
            link=link, expires=date(expires), topic_list=topic_list)
        from_email = settings.VOLUNTEER_FROM_EMAIL

        email_data = []

        for email in emails:
            # Append multiple email data items, otherwise volunteers could
            # see each other's email addresses
            email_data.append(
                (subject, body, from_email, [email]))

        try:
            send_mass_mail(email_data)
            num_emails = len(email_data)
            pluralize = ['s', ''][num_emails == 1]
            messages.info(request, '{0} volunteer email{1} sent.'.format(
                num_emails, pluralize))
        except Exception as err:
            messages.error(request, "Unable to send emails. {0}".format(err))

