"""Views"""
from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required, permission_required
from django.contrib import messages
from django.conf import settings
from django.core.urlresolvers import reverse

from volunteer.models import Volunteer, VolunteerTopic
from volunteer.forms import VolunteerTopicForm, UserForm
from usta.models import get_site_settings


def view_volunteer_topics(request):
    """View Volunteer Topics

    Only the following people can view this page:
    - superusers
    - volunteers
    - volunteer topic editors

    If the user does not have permission, we redirect elsewhere with a message
    saying they are not logged in as a user. We do not respond with
    403-Forbidden, because this view is available by url in the templates.

    Volunteers can update their subscriptions here as well
    """
    user = None
    if request.user.is_authenticated():
        user = request.user
    topics = VolunteerTopic.objects.all()
    volunteers = Volunteer.objects.filter(user=user)
    if (not volunteers.exists() and not request.user.is_superuser and
        not request.user.has_perm('volunteer.change_volunteertopic')):
        error_msg = "You are not logged in as a volunteer."
        site_settings = get_site_settings()
        node = site_settings.get_setting('volunteer_node', None)
        location = 'usta:index'  # Redirect home if no node
        if node and node.path:
            error_msg += " See below for how to become one."
            location = reverse('htmleditor:view', args=(node.path,))
        messages.error(request, error_msg)
        return redirect(location)
    memberships = [volunteer.topic.pk for volunteer in volunteers]
    subscriptions = []
    for volunteer in volunteers:
        if volunteer.subscribed:
            subscriptions.append(volunteer.topic.pk)

    if request.method == 'POST':
        for volunteer in volunteers:
            key = 'subscribe-{0}'.format(volunteer.topic.pk)
            volunteer.subscribed = request.POST.get(key, False)
            volunteer.save()
        return redirect('volunteer:view_volunteer_topics')

    return render(request, 'volunteer/view_volunteer_topics.html', {
        'topics': topics,
        'memberships': memberships,
        'subscriptions': subscriptions,
        'title': 'Volunteer Topics',
        'show_generic': True,
    })


@permission_required('volunteer.change_volunteertopic', raise_exception=True)
def edit_volunteer_topic(request, topic_pk):
    """Edit Volunteer Topic

    Only superusers or volunteer topic editors can use this view,
    otherwise 403-Forbidden is returned.

    Only Topic name and content can be edited here, use the admin to edit
    actual volunteers for a topic.
    """
    topic = get_object_or_404(VolunteerTopic, pk=topic_pk)

    if request.method == 'POST':
        form = VolunteerTopicForm(request.POST, instance=topic)
        if form.is_valid():
            form.save()
            return redirect('volunteer:view_volunteer_topics')
    else:
        form = VolunteerTopicForm(instance=topic)

    return render(request, 'volunteer/edit_volunteer_topic.html', {
        'form': form,
        'topic': topic,
        'title': topic.name,
        'show_generic': True,
    })


def new_volunteer(request):
    if request.method == 'POST':
        form = UserForm(request.POST)
        if form.is_valid():
            #print("Form is valid!")
            code = form.cleaned_data['code']
            user = form.save()
            for topic in code.topics.all():
                topic.volunteers.create(user=user)
            return redirect(settings.LOGIN_URL)
    else:
        code = request.GET.get('code', '')
        form = UserForm(initial={
            'code': code,
        })
    return render(request, 'volunteer/new_volunteer.html', {
        'form': form,
    })
