"""VolunteerCode"""
import uuid
import datetime
import sys

from django.db import models
from django.conf import settings
from django.utils import timezone
from django.db.models.signals import pre_save
from django.dispatch import receiver
from django.core.urlresolvers import reverse
from django.core import mail
from django.core.mail.message import EmailMultiAlternatives
from django.contrib.sites.models import Site

from ckeditor_uploader.fields import RichTextUploadingField


MAX_LENGTH = 100


class VolunteerTopicManager(models.Manager):
    """Simple Manager for Document"""
    def get_by_natural_key(self, name):
        """Get Document by name"""
        return self.get(name=name)


class VolunteerTopic(models.Model):
    objects = VolunteerTopicManager()

    name = models.CharField(max_length=MAX_LENGTH, unique=True)
    content = RichTextUploadingField()
    last_modified = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ['name']
        verbose_name = "Volunteer Topic"

    def num_volunteers(self):
        return self.volunteers.count()

    def num_subscribed(self):
        return self.volunteers.filter(subscribed=True).count()

    def natural_key(self):
        """A human readable unique key"""
        return (self.name,)

    def __str__(self):
        return self.name


class Volunteer(models.Model):
    user = models.ForeignKey(settings.AUTH_USER_MODEL, models.CASCADE,
                             related_name='+')
    topic = models.ForeignKey('VolunteerTopic', models.CASCADE,
                              related_name='volunteers')
    subscribed = models.BooleanField(
        blank=True, default=False,
        help_text="Receive emails when this topic is updated?")

    class Meta:
        unique_together = (('user', 'topic'),)

    def __str__(self):
        return str(self.user)


class VolunteerCode(models.Model):
    """One time code to invite a volunteer"""
    code = models.UUIDField(primary_key=True, default=uuid.uuid4,
                            editable=False)
    expired = models.BooleanField(blank=True, default=False)
    date_created = models.DateField(auto_now_add=True)
    expires_after = models.PositiveSmallIntegerField(
        default=7,
        help_text="How many days is this code valid?")
    topics = models.ManyToManyField(VolunteerTopic)

    def valid(self):
        """Returns whether this code is not expired

        This also triggers the check for expires_after, but does not actually
        save. Saving must be done explicitly after."""
        if self.date_created:
            delta = datetime.date.today() - self.date_created
            if delta.days > self.expires_after:
                self.expired = True
        return not self.expired
    valid.boolean = True

    def __str__(self):
        return '{0} - {1}'.format(self.code, self.date_created)


VOLUNTEER_UPDATE_MESSAGE = """Hello {name},<br>
<br>
The <a href="{topic_link}">{topic_name} Volunteer Topic</a> has been updated.<br>
<br>
<a href="{subscription_link}">Change subscriptions</a><br>
<br>
<small>1200 Academy Street • Kalamazoo College • Kalamazoo, MI 49006 • Phone: (269) 337-7343 • Fax: (269) 337-7385</small>
"""


@receiver(pre_save, sender=VolunteerTopic)
def on_volunteer_presave(sender, instance, **kwargs):
    #print(Site.objects.get_current().domain)
    # Ignore creation on topic
    if not instance.last_modified:
        return
    if not getattr(settings, 'VOLUNTEER_FROM_EMAIL', None):
        # Can't send emails if no from email
        return
    delta = timezone.now() - instance.last_modified
    if delta.seconds / 60 >= settings.EMAIL_LIMIT_MINUTES:
        #print("Sending email...")
        subject = "USTA Boys {0} Update".format(instance.name)
        from_email = settings.VOLUNTEER_FROM_EMAIL
        emails = []
        for volunteer in instance.volunteers.filter(subscribed=True):
            #print("-"*80)
            user = volunteer.user
            name = user.get_full_name() or user.get_username()
            #print("Working on " + name)
            if not user.is_active:
                #print("Ignoring inactive user")
                continue
            if not user.email:
                #print("Ignoring user without email")
                continue
            link = "http://{0}{1}".format(
                Site.objects.get_current().domain,
                reverse('volunteer:view_volunteer_topics'))
            message = VOLUNTEER_UPDATE_MESSAGE.format(
                name=name,
                topic_name=instance.name,
                topic_link=link,
                subscription_link=link)
            #print("Subject: [{0}]".format(subject))
            #print("Message:\n{0}".format(message))
            email = EmailMultiAlternatives(subject, message, from_email,
                                           [user.email])
            email.attach_alternative(message, "text/html")
            emails.append(email)
        if emails:
            try:
                connection = mail.get_connection()
                connection.open()
                connection.send_messages(emails)
                connection.close()
            except:
                sys.stderr.write("Unable to send email, is it configured?\n")
    #else:
    #    print("Email already sent earlier, ignoring...")


@receiver(pre_save, sender=VolunteerCode)
def on_volunteer_code_presave(sender, instance, **kwargs):
    "Make sure that an expire check is triggered"
    del sender, kwargs  # Not used
    instance.valid()  # Trigger setting .expired, will be saved
