"""tests.py"""
from django.test import TestCase
from django.contrib.auth.models import User
from django.core.urlresolvers import reverse

from sponsors.models import Company, get_random_sponsor
from htmleditor.models import Image, Banner

HISTOGRAM_ITERATIONS = 500
TEST_ITERATIONS = 3


class RandomSponsorTestCase(TestCase):
    """Tests for randomness of get_random_sponsor"""
    def setUp(self):
        """Create a company for each level in order of best level"""
        self.company_a = Company.objects.create(
            name="Company A",
            sponsor_level=Company.LEVEL_SPONSOR_TITLE_EXHIBITION,
            image='test',
            link='www.example.com')
        self.company_b = Company.objects.create(
            name="Company B",
            sponsor_level=Company.LEVEL_SPONSOR_SPECIAL_EVENT,
            image='test',
            link='www.example.com')
        self.company_c = Company.objects.create(
            name="Company C",
            sponsor_level=Company.LEVEL_SPONSOR_DISTINGUISHED,
            image='test',
            link='www.example.com')
        self.company_d = Company.objects.create(
            name="Company D",
            sponsor_level=Company.LEVEL_CORP_PATRON_DISTINGUISHED,
            image='test',
            link='www.example.com')
        self.company_e = Company.objects.create(
            name="Company E",
            sponsor_level=Company.LEVEL_CORP_PATRON_STANDARD,
            image='test',
            link='www.example.com')
        image = Image.objects.create(
            image='test',
            alt_text='test')
        for company in [
            self.company_a,
            self.company_b,
            self.company_c,
            self.company_d,
            self.company_e]:
            banner = Banner.objects.create(
                name=company.name,
                image=image,
                company_sponsor=company)

        self.user = User.objects.create_user(
            'testuser', password='pass', email='test@example.com')

    def _histogram_test(self):
        """One iteration of checking sponsor frequency"""
        histogram = {}
        # pylint: disable=unused-variable
        for i in range(HISTOGRAM_ITERATIONS):
            company = get_random_sponsor()
            if company.pk in histogram:
                histogram[company.pk] += 1
            else:
                histogram[company.pk] = 1

        # Debug print
        #print("Histogram Breakdown out of {0}".format(HISTOGRAM_ITERATIONS))
        #for pk in histogram:
        #    company = Company.objects.get(pk=pk)
        #    print("{0}: {1}".format(company.name, histogram[pk]))

        success = True

        if histogram[self.company_a.pk] < histogram[self.company_b.pk] or \
           histogram[self.company_b.pk] < histogram[self.company_c.pk] or \
           histogram[self.company_c.pk] < histogram[self.company_d.pk] or \
           histogram[self.company_d.pk] < histogram[self.company_e.pk]:
            success = False

        return success

    def test_histogram(self):
        """Assert that random sponsors are properly weighted

        We run the test a couple times, in case of the rare chance one fails.
        """
        success = False
        # pylint: disable=unused-variable
        for i in range(TEST_ITERATIONS):
            if self._histogram_test():
                success = True
                break

        self.assertTrue(success)

    def test_view_sponsors(self):
        """Test the view view_sponsors returns 200 OK"""
        self.client.force_login(self.user)
        url = reverse('sponsors:view_sponsors')
        response = self.client.get(url)
        self.assertEqual(response.status_code, 200,
                         "View did not return a 200.")
