"""models.py"""
import random

from django.db import models
from django.utils.html import format_html

MAX_LENGTH = 100


class Company(models.Model):
    """Represents a Company that supports USTA

    A Company may give finanical support and/or advertise with USTA Boys"""
    LEVEL_NONE = 0
    LEVEL_SPONSOR_TITLE_EXHIBITION = 1
    LEVEL_SPONSOR_SPECIAL_EVENT = 2
    LEVEL_SPONSOR_DISTINGUISHED = 3
    LEVEL_CORP_PATRON_DISTINGUISHED = 4
    LEVEL_CORP_PATRON_STANDARD = 5
    LEVELS = (
        (LEVEL_NONE, "None"),
        (LEVEL_SPONSOR_TITLE_EXHIBITION, "Title & Exhibition Sponsor"),
        (LEVEL_SPONSOR_SPECIAL_EVENT, "Special Event Sponsor"),
        (LEVEL_SPONSOR_DISTINGUISHED, "Distinguished Sponsor"),
        (LEVEL_CORP_PATRON_DISTINGUISHED, "Distinguished Corporate Patron"),
        (LEVEL_CORP_PATRON_STANDARD, "Corporate Patron"),
    )

    name = models.CharField(max_length=MAX_LENGTH, unique=True)
    image = models.ImageField(upload_to="companies/", blank=True)
    link = models.URLField(blank=True)
    sponsor_level = models.PositiveSmallIntegerField(choices=LEVELS, default=0)
    advertiser = models.BooleanField(blank=True, default=False)

    class Meta:
        verbose_name_plural = "companies"
        ordering = ['name']

    def admin_image(self):
        """Display image for admin interface"""
        html = ''
        if self.image:
            # pylint: disable=no-member
            html = format_html('<img height="64px" src="{0}">', self.image.url)
        return html
    admin_image.short_description = "Image"

    def __str__(self):
        return self.name


class Individual(models.Model):
    """Represents an Individual who donated"""
    name = models.CharField(max_length=MAX_LENGTH)

    def __str__(self):
        return self.name


def get_random_sponsor():
    """Return a random sponsor

    Those with better (lower) sponsor_levels appear more often."""
    companies = Company.objects.all().exclude(sponsor_level=0).exclude(
        banner=None)
    choices = []
    num_levels = len(Company.LEVELS)
    for company in companies:
        # pylint: disable=unused-variable
        for i in range(num_levels - company.sponsor_level):
            choices.append(company.pk)
    if choices:
        chosen_pk = random.choice(choices)
        return Company.objects.get(pk=chosen_pk)
    else:
        return None
